<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once '../helpers/helpers.php';

// Set active page for navigation
$activePage = 'suppliers';

// Check if user is logged in
requireLogin();

// Check if user has permission to manage suppliers
if (!hasPermission('manage_suppliers')) {
    echo json_encode([
        'success' => false,
        'message' => "You don't have permission to edit suppliers."
    ]);
    exit;
}

// Get supplier ID from URL
$supplier_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

if (!$supplier_id) {
    echo json_encode([
        'success' => false,
        'message' => "Invalid supplier ID."
    ]);
    exit;
}

// Get supplier details
$stmt = $pdo->prepare("SELECT * FROM suppliers WHERE id = ?");
$stmt->execute([$supplier_id]);
$supplier = $stmt->fetch();

if (!$supplier) {
    echo json_encode([
        'success' => false,
        'message' => "Supplier not found."
    ]);
    exit;
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $name = trim($_POST['name']);
        $contact_person = trim($_POST['contact_person']);
        $phone = trim($_POST['phone']);
        $email = trim($_POST['email']);
        $address = trim($_POST['address']);
        $tax_number = trim($_POST['tax_number']);

        if (empty($name)) {
            throw new Exception("Supplier name is required");
        }

        $stmt = $pdo->prepare("
            UPDATE suppliers 
            SET name = ?, 
                contact_person = ?, 
                phone = ?, 
                email = ?, 
                address = ?, 
                tax_number = ? 
            WHERE id = ?
        ");
        $stmt->execute([$name, $contact_person, $phone, $email, $address, $tax_number, $supplier_id]);

        echo json_encode([
            'success' => true,
            'message' => "Supplier updated successfully!"
        ]);
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'message' => $e->getMessage()
        ]);
    }
    exit;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Edit Supplier - POS System</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        body {
            background: #f8fafc;
        }
        .card {
            border: none;
            border-radius: 15px;
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
        }
        .form-label {
            font-weight: 500;
            color: #374151;
        }
        .form-control {
            border-radius: 8px;
            border: 1px solid #e5e7eb;
            padding: 0.6rem 1rem;
        }
        .form-control:focus {
            border-color: #3b82f6;
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }
        .btn-primary {
            background-color: #3b82f6;
            border: none;
            padding: 0.6rem 1.5rem;
            border-radius: 8px;
            font-weight: 500;
        }
        .btn-primary:hover {
            background-color: #2563eb;
        }
        .btn-secondary {
            background-color: #6b7280;
            border: none;
            padding: 0.6rem 1.5rem;
            border-radius: 8px;
            font-weight: 500;
        }
        .btn-secondary:hover {
            background-color: #4b5563;
        }
        .required-field::after {
            content: "*";
            color: #ef4444;
            margin-left: 4px;
        }
    </style>
</head>
<body>
    <div class="container-fluid py-4">
        <div class="row">
            <?php include '../includes/sidebar.php'; ?>
            <div class="col-md-10 ms-sm-auto px-md-4">
                <?php include '../includes/topbar.php'; ?>

                <!-- Header -->
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <div>
                        <h2 class="mb-1">Edit Supplier</h2>
                        <p class="text-muted mb-0">Update supplier information</p>
                    </div>
                    <a href="suppliers.php" class="btn btn-secondary">
                        <i class="fas fa-arrow-left me-2"></i>Back to Suppliers
                    </a>
                </div>

                <!-- Main Content -->
                <div class="card">
                    <div class="card-body p-4">
                        <?php if (isset($error)): ?>
                            <div class="alert alert-danger"><?php echo $error; ?></div>
                        <?php endif; ?>

                        <form method="post" class="needs-validation" novalidate>
                            <div class="row g-3">
                                <div class="col-md-6">
                                    <label for="name" class="form-label required-field">Supplier Name</label>
                                    <input type="text" class="form-control" id="name" name="name" 
                                           value="<?php echo htmlspecialchars($supplier['name']); ?>" required>
                                    <div class="invalid-feedback">Please enter supplier name.</div>
                                </div>

                                <div class="col-md-6">
                                    <label for="contact_person" class="form-label">Contact Person</label>
                                    <input type="text" class="form-control" id="contact_person" name="contact_person"
                                           value="<?php echo htmlspecialchars($supplier['contact_person']); ?>">
                                </div>

                                <div class="col-md-6">
                                    <label for="phone" class="form-label">Phone</label>
                                    <input type="tel" class="form-control" id="phone" name="phone"
                                           value="<?php echo htmlspecialchars($supplier['phone']); ?>">
                                </div>

                                <div class="col-md-6">
                                    <label for="email" class="form-label">Email</label>
                                    <input type="email" class="form-control" id="email" name="email"
                                           value="<?php echo htmlspecialchars($supplier['email']); ?>">
                                </div>

                                <div class="col-md-6">
                                    <label for="address" class="form-label">Address</label>
                                    <textarea class="form-control" id="address" name="address" rows="3"><?php echo htmlspecialchars($supplier['address']); ?></textarea>
                                </div>

                                <div class="col-md-6">
                                    <label for="tax_number" class="form-label">Tax Number</label>
                                    <input type="text" class="form-control" id="tax_number" name="tax_number"
                                           value="<?php echo htmlspecialchars($supplier['tax_number']); ?>">
                                </div>
                            </div>

                            <div class="mt-4">
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-save me-2"></i>Save Changes
                                </button>
                                <a href="suppliers.php" class="btn btn-secondary ms-2">Cancel</a>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Form validation
        (function () {
            'use strict'
            var forms = document.querySelectorAll('.needs-validation')
            Array.prototype.slice.call(forms).forEach(function (form) {
                form.addEventListener('submit', function (event) {
                    if (!form.checkValidity()) {
                        event.preventDefault()
                        event.stopPropagation()
                    }
                    form.classList.add('was-validated')
                }, false)
            })
        })()
    </script>
    <?php include '../includes/footer.php'; ?>
</body>
</html> 